function S = chris_detect_spikes(S)
% S = CHRIS_DETECT_SPIKES(S)
%
% If called without an argument, this function acts as a script. It reads certain 'hard-wired' ABF files 
% one by one, and looks for spikes in them. It then returns the result as a bunch of coordinated tables 
% within the structure S. This way it works with oscillations (bumps) of different frequencies.
%
% If called with an argument S, it skips reading and spike-finding, and goes directly to high-level
% processing.
%
% To see a figure with spikes, go down to SENSOR subroutine and there put if(1) for the figure-building block.

% Oct 23 2012: Revision.
% Oct 97 2013: Redirected to new folders.
% Feb 26 2014: Now it gathers some important statistics and reports it by neuron type.
% May 12 2014: Repurposed for new set of data.
% May 14 2014: Troubleshooting spike detection algorithms to work with new data. Traces are now normalized to [-60 10]
%               spread, and then processed with moderately conservative thresholding values.
% May 23 2014: Adjusting for amplitude-injections.
% Jan 17 2015: Revisited; very mild changes in code usability.

doAmpsNotFreq = 0;      % Set to 1 is you need to process amplitude-bumps. Set to 0 for freq-bumps.

if(nargin<1)        
    S.id = [];      S.animal = [];  S.cell = [];    
    S.lat = [];     S.amps = [];    S.group = [];   S.sweep = [];   % Here everything will be stored
    
    fileName = 'C:\Users\Arseny\Documents\4_Cells Classification\2014 data\BumpsKey.xlsx';
    [dataN,dataT] = xlsread(fileName, 1);   % Numbers and text
    dataT = dataT(2:end,2:end);             % First line contains headers, and first column is empty
    
    sn = 1:size(dataN,1);           % Debugging tool. You may replace this value by something else (below) if you need to shorten the list of trials to be processed.
    %sn = [1 129];  
    % sn = [44 50] + 50;
    % sn = (1:25) + 100;
    % sn = 1:10;
    dataN = dataN(sn,:); dataT = dataT(sn,:);     
    %%%% If you need a debugging figure, Ctrl-F for "meow" below, and change 0 to 1. Change to 0 to disable it.

    nFiles = size(dataN,1);         % how many files to read
    for(q=1:nFiles)
        id = dataN(q);
        if(doAmpsNotFreq)
            file2read = dataT{q,2};	% Filename
            S = enjoy(S,id,2,['C:\Users\Arseny\Documents\4_Cells Classification\2014 data\Raw Bumps files\' file2read '.abf']);            % Main processing
        else
            file2read = dataT{q,1};	% Filename
            S = enjoy(S,id,1,['C:\Users\Arseny\Documents\4_Cells Classification\2014 data\Raw Bumps files\' file2read '.abf']);            % Main processing
        end        
    end

%     fprintf('%5s\t%5s\t%5s\t%9s\t%9s\t%5s\n','Tad','Cell','Group','Amp mV','Lat ms','Sweep'); % Screen
%     fprintf('%5d\t%5d\t%5d\t%9.2f\t%9.2f\t%5d\n',[S.animal S.cell S.group S.amps   S.lat/10     S.sweep]');

    if(doAmpsNotFreq)
        fprintf('Amplitudes. Writing raw data... ');
        fid = fopen('C:\Users\Arseny\Documents\4_Cells Classification\2014 data\chris_bumps_amps_dat.txt','w');    
        % fprintf(fid,'%9s\t%5s\t%9s\t%9s\t%5s\r\n','id','Group','Amp mV','Lat ms','Sweep');
        % fprintf(fid,'%9d\t%5d\t%9.2f\t%9.2f\t%5d\r\n',[S.id S.group S.amps   S.lat/10     S.sweep]');   
    else
        fprintf('Frequencies. Writing raw data... ');
        fid = fopen('C:\Users\Arseny\Documents\4_Cells Classification\2014 data\chris_bumps_freq_dat.txt','w');    
        fprintf(fid,'%9s\t%5s\t%9s\t%9s\t%5s\r\n','id','Group','Amp mV','Lat ms','Sweep');
        fprintf(fid,'%9d\t%5d\t%9.2f\t%9.2f\t%5d\r\n',[S.id S.group S.amps   S.lat/10     S.sweep]');             
    end
    fclose(fid);
    fprintf('Done.\n');
end     % Now we certainly have S: either it was received, or it was read 

if(doAmpsNotFreq)
    process(S,2);
else
    process(S,1);
end    

end



function S = enjoy(S,id,mode,fileName)
% The working horse function of this script.
% MODE is 1 for frequencies, and 2 for amplitudes.

areas =      [1   4250 8300 12300 16400 19600];     % Limits of current injection bouts
bumpStarts = [400 4400 8400 12400 16400] + 40;      % Where each bump started, estimated by eye. The actual injection starts at about +14
safetyTail = 400;                                   % Approximate delay between the start of the bout and the actual response (passive electric properties?)

if(mode==1)
    boutLength = [2000 2000 1800 2000 2000];            % Length of each bout (one of them is a bit different)
else
    boutLength = [2000 2000 2000 2000 2000];
end

try
    temp = abfload(fileName);
catch
    fprintf('Reading of file %d failed\n',id);
    return;
end

data = squeeze(temp(:,1,:));        % Potential, mV
% current = squeeze(temp(:,2,:));     % Current being injected, pA

% figure; set(gcf,'Color','w'); hold on; plot(data,'g-'); plot(areas,ones(size(areas))*data(1),'ks'); hold off; error('Ha-ha!');   % Debugger figure

data = data-min(data(:));   % My new idea is that maybe we should re-normalize all data, just to have the threshold values consistent                  
data = data/max(data(:));
data = data*70 - 60;    


for(iSweep=1:size(data,2))      % A typical file contains 10 sweeps
    [lat,amps] = sensor(data(:,iSweep),iSweep,id);
    group = nan(size(lat));     % "Which bout was it" data will be stored here
    
    for(iGroup = 1:5)           % Bout of bump-injections
        group((lat>=areas(iGroup)) & (lat<areas(iGroup)+boutLength(iGroup)+safetyTail)) = iGroup;  % Which bout was it
        lat(group==iGroup) = lat(group==iGroup) - bumpStarts(iGroup);                   % Latency adjustment (translating from "from the beginning" to "within the bump group")        
    end
    
    % dispf([(1:length(lat))' lat group]); % ---- Debugger
    g = ~isnan(group(:));      % Good (pocketed) spikes
        
    S.lat   = [S.lat;   lat(g)];
    S.amps  = [S.amps;  amps(g)];
    S.group = [S.group; group(g)];
    S.sweep = [S.sweep; iSweep*ones(sum(g),1)];
%     S.animal= [S.animal;   animal*ones(length(lat),1)];
%     S.cell  = [S.cell;     cell*ones(length(lat),1)];
    S.id    = [S.id; id*ones(sum(g),1)];    
end

%title(num2str(id)); % Presumably there's a figure somewhere there

end



function process(S,mode)
% High-order processing of latency information: 
% it calculates the average number of spikes that each neuron had per bump of current injection.
% MODE is 1 for frequencies, and 2 for amplitudes.

if(mode==1)
    nBumps =     [20 10 6 5 4];                         % N Bumps in each injection bout
    bumpLength = [2000 2000 1800 2000 2000]./nBumps;    % 200 ms, sampled at 10 kHz
elseif(mode==2)
    nBumps =     [6 6 6 6 6];                       
    bumpLength = [1800 1800 1800 1800 1800]./nBumps;    % 200 ms, sampled at 10 kHz
end
bumpShift = [0 cumsum(nBumps)];                         % To shift from local to global bump number (like enumerating across all bumps in all groups)
bumpStarts = [400 4400 8400 12400 16400] + 40;          % Only needed for plotting purpuses (below).

bag = [];

idList = unique(S.id);

if(length(idList)<2)
    return;                         % No need ot make a summary figure if there's only one cell processed
end
for(idCount=1:length(idList))                                                   % For each cell
    id = idList(idCount);
    g  = (S.id==id);                                                            % Spikes of this cell
    %%% figure; hold on; plot(S.lat(g),randn(size(S.lat(g))),'.'); hold off; error('Ha-ha'); % Debugging figure
    bump = bumpShift(S.group(g))' + ceil(S.lat(g)./(bumpLength(S.group(g))'));  % What bump number (of all recorded) it was
    bumpStats = zeros(1,sum(nBumps));
    nSweeps = max(S.sweep(g));
    for(q=1:sum(nBumps))                                                        % For each bump - count the spikes.
        bumpStats(q) = sum(bump==q);
    end
    fprintf('%8d\t',id); dispf(bumpStats/nSweeps,'%3.1f');    
    bag = [bag; bumpStats/nSweeps];
end

xData = [];                         % Calculate where the bumps are located (approximately, for plotting)
for(iBout = 1:length(nBumps))
    xData = [xData; bumpStarts(iBout)+(0:nBumps(iBout)-1)'*bumpLength(iBout)];
end

figure; hold on;                    % Average number of spikes per bump
plot(xData,bag','g-');
plot(xData,mean(bag),'.-k');
hold off;

end



function [lat,amps] = sensor(v,iSweep,id)
% This function actually thresholds the spikes. The logic it follows may be
% somewhat confusing, so 

width = 20;                 % Target spike width (the "neck point" will be set at a level at which a spike reaches this width)
% width2 = 100;             % How much to the left a minimum may lie from a respective peak-detection point
iflectionThreshold = 0.01;  % Threshold for inflection points.
peakThreshold = 2;        % Treshold for peak size, mV (from the Neck point)

%%% It will take all narrow high peaks (according to PEAKTHRESHOLD) and all peaks that have an inflection (based on IFLECTIONTHRESHOLD)

[fb,fa] = butter(3,0.2);
v = myfilt(fb,fa,v);

f = v(width:end)-v(1:end-width+1);            % If v has same value WIDTH points to the right
necks = find((f(1:end-1)>0)&(f(2:end)<0));    % Roots: Either a peak, or a valley
shape = v(necks+floor(width/2))-v(necks);     % Value in the middle
necks = necks(shape>0);                       % Only peaks

d1 = diff(v,1);     d1 = [d1([1  ]); d1(:)];
d2 = diff(v,2);     d2 = [d2([1 1]); d2(:)];
inflections = max(0,d1).*max(0,d2);                     % A bit of d1 is added to d2 for higher contrast
infPoints = find((inflections(2:end)>=iflectionThreshold)&(inflections(1:end-1)<iflectionThreshold)); % Inflection points (only positive)
locMin = find((d1(2:end)>0)&(d1(1:end-1)<=0));  % Local minima

pMax = nan(size(necks));
pLeft = pMax;	pNearestInflection = pMax;	pMinLeft = pMax;	pMinRight = pMax;

for(iPoint = 1:length(necks)) % For each potential peak
    pMax(iPoint) = necks(iPoint)-1+find(v(necks(iPoint)+(0:width))==max(v(necks(iPoint)+(1:width))));  % Nearest local maximum to the right
    if(necks(iPoint)>=locMin(1))                                                % If there are any minimi to the left from our potential peak
        pMinLeft(iPoint) = max(locMin(locMin<=necks(iPoint)));                  % Nearest local minimum to the left
    else
        pMinLeft(iPoint) = necks(iPoint);
    end
    
    if(necks(iPoint)<=locMin(end))                                              % If there are any minimi to the right        
        pMinRight(iPoint) = min(locMin(locMin>=necks(iPoint)));                 % Nearest local minimum to the right          
    else
        pMinRight(iPoint) = necks(iPoint);
    end
    
    if(~isempty(infPoints))                                                             % If there are any inflection points
        if(necks(iPoint)>=infPoints(1))
            pNearestInflection(iPoint) = max(infPoints(infPoints<=necks(iPoint)));      % Nearest inflection
        else
            pNearestInflection(iPoint) = 1;
        end
    end
    
    %if(pMinLeft(iPoint)>=pNearestInflection(iPoint))                                 % Choose between local minimum and inflection
    %    pLeft(iPoint) = pMinLeft(iPoint);
    %else
    %    pLeft(iPoint) = pNearestInflection(iPoint);
    %    %%% The problem with this approach is that depending on the unpredictable thresholding some spikes in a train get measured either from the
    %    %%% infliction point, or from a valley. It's not a healthy way to do it, as this way I can not measure decensitization.
    %end
    pLeft(iPoint) = pMinLeft(iPoint);
end

if(isempty(pNearestInflection))
    pNearestInflection = ones(size(pMax));  % Just to make formulas below work, even though this is an obvious crutch and doesn't make any sense
end

%goodIs = ((v(pMax(:))-v(pLeft(:)))>=peakThreshold) & ((v(pMax(:))-v(pMinRight(:)))>=peakThreshold);    % Doesn't work. Accepts oscillations as well.
goodIs = ( (v(pMax(:))-v(necks(:)))>=peakThreshold ) | ( pNearestInflection(:)>=pMinLeft(:) ); % Let's take all narrow peaks & all peaks after thresholded inflictions.
% It's hard to detect both peaks in the high-frequency train, and 2nd peaks in the slow-frequency train.,
% But this way everything seems to be detected, more or less.

necks = necks(goodIs);
pMax  = pMax(goodIs);
pLeft = pLeft(goodIs);

if(0) % Most nice and useful debugging figure -=-=-=--=-=-=-=-=-=-=-=-=-=-=-=-=-=-=- meow
    areas =      [1   4250 8300 12300 16400 19600];     % Limits of injection blocks
    if(iSweep==1)    % Good indicative figure
        figure;
        subplot(4,1,4); hold on;    % Lower section (tresholding)
        plot(inflections,'g-');
        plot(infPoints,iflectionThreshold*ones(size(infPoints)),'b.');        
        hold off;
        
        subplot(4,1,1:3); hold on;  % Higher section (results)
        plot(1:length(v),v,'g-');
        plot(necks,v(necks),'r.');
        plot(pMax,v(pMax),'r+');
        plot(pLeft,v(pLeft),'b.');
        plot(infPoints,v(infPoints),'g.');
        
        for(iBout = 1:length(areas)-1)
            nSpikes = sum((pMax>areas(iBout) & (pMax<=areas(iBout+1))));
            yLim = get(gca,'YLim');
            text(mean(areas(iBout + (0:1))),yLim(1)+5,num2str(nSpikes),'HorizontalAlign','Center','Color','b');
        end
        
        hold off;
        % legend('Voltage','Necks','Max','Min','Inflections','Location','SouthEast');
        ylabel(id);
        drawnow;
        % error('=)');
    end
end

lat = pMax;  % I used pLeft initially, but for many peaks (especially for the 1st peaks within each bout) I put it too far on the left.
amps = v(pMax)-v(pLeft);

end



function x2 = myfilt(b,a,x)
% Filtering without ripples at the end
x2 = filtfilt(b,a,[x(:); ones(length(b),1)*mean(x(end-10:end))]);
x2 = x2(1:length(x));
end
